/***********************************************************************/
/*                                                                     */
/*  File :  DPBAPI.h                                                   */
/*                                                                     */
/*  FUJIFILM ASK-1500 / ASK-2000 / ASK-4000 / ASK-4000A	/ ASK-2500	   */
/*	series Printer Control DLL										   */
/*                                                                     */
/*      (DLL public Interfaces header file)                            */
/*                                                                     */
/*      Last Modified at 2008/10/09 for DLL Ver.2.6.8                  */
/*                                                                     */
/*  Copyright (C) 2001-2008 FUJIFILM Corporation		               */
/*                                                                     */
/***********************************************************************/


#ifndef __DPBAPI_H__
#define __DPBAPI_H__


#include <windows.h>


#ifdef __cplusplus
extern "C" {
#endif


/************************************************/
/*                                              */
/*  Constants definition                        */
/*                                              */
/************************************************/


/* Printer paper box count  */
#define MCP_PAPERBOX_COUNT                  1


/* Can open printer number at the same time */
#define MCP_MAX_OPEN_PRINTER                8



/************************************************/
/*                                              */
/*  Data type definition                        */
/*                                              */
/************************************************/


/* Printer device information   */
typedef struct {
    short       nUsbNo;         /* USB connection number        */
    short       reserved;
    short       nPrinterId;     /* Printer product type ID      */
    short       nMediaType;     /* Media type in printer now    */
    char        szSerialNumber[64];     /* Printer serial number    */
} MCP_PRINTER_INFO, *PMCP_PRINTER_INFO;

    /* For nPrinterId parameter     */
	#define	MCP_PRINTER_ID_DPB1000			(0)		/* Add 2005/08/24 Ver.1.6.0 */
    #define MCP_PRINTER_ID_DPB1500          (2)
    #define MCP_PRINTER_ID_ASK1500          (2)     /* Add 2005/06/08 Ver.1.3.0 */
    #define MCP_PRINTER_ID_DPB4000          (4)     /* Add 2005/07/12 Ver.1.4.3 */
    #define MCP_PRINTER_ID_ASK4000          (4)     /* Add 2005/12/22 Ver.2.0.2 */
    #define MCP_PRINTER_ID_DPB6000          (6)     /* Add 2005/12/20 Ver.2.0.1 */
    #define MCP_PRINTER_ID_ASK2000          (6)     /* Add 2005/12/26 Ver.2.0.3 */
    #define MCP_PRINTER_ID_ASK4000A         (9)     /* Add 2007/06/29 Ver.2.5.0 */
    #define MCP_PRINTER_ID_DPB4000A         (9)     /* Add 2007/06/29 Ver.2.5.0 */
	#define MCP_PRINTER_ID_ASK2500			(11)	/* Add 2008/10/09 Ver.2.6.8 */
	#define MCP_PRINTER_ID_DPB2500			(11)	/* Add 2008/10/09 Ver.2.6.8 */
    #define MCP_PRINTER_ID_UNKNOWN          (-1)

    /* For nMediaType parameter (Ink ribbon type in printe now) */
    #define MCP_MEDIA_TYPE_UNKNOWN          (-1)    /* Printer not printed after turn on.   */
    #define MCP_MEDIA_TYPE_NONE             (0)     /* No ink ribbon.                       */
    #define MCP_MEDIA_TYPE_L                (1)
    #define MCP_MEDIA_TYPE_K                (2)
    #define MCP_MEDIA_TYPE_2L               (3)
    #define MCP_MEDIA_TYPE_6X8_A5           (4)
	/* For ASK-4000A */
    #define MCP_MEDIA_TYPE_A4		        (5)    // Add for ASK-4000A 2007/06/29 addition Ver.2.5.0
	/* For ASK-4000 */
    #define MCP_MEDIA_TYPE_8X10             (6)    /* Modify 2005/11/21 Ver.1.7.2 */
    #define MCP_MEDIA_TYPE_8X12             (7)    /* Add 2005/12/20 Ver.2.0.1 */
 

/* Paper information    */
typedef struct {
    POINT       PaperSize;      /* Print paper size ( x : width [pixel], y : height [pixel] )     */
} MCP_PAPERINFO, *PMCP_PAPERINFO; 

    /* For PaperSize.x parameter    */
    #define MCP_PAPER_WIDTH_L               1536
    #define MCP_PAPER_WIDTH_K               1832
    #define MCP_PAPER_WIDTH_2L              1536
    #define MCP_PAPER_WIDTH_6X8             1832
    #define MCP_PAPER_WIDTH_A5              1832	// For ASK-2000
    #define MCP_PAPER_WIDTH_8X10            2444    /* Add 2005/07/12 Ver.1.4.3 */
    #define MCP_PAPER_WIDTH_8X12            2444    /* Add 2005/12/20 Ver.2.0.1 */
    #define MCP_PAPER_WIDTH_A4				2528    // Add for ASK-4000A 2007/06/29 addition Ver.2.5.0
    #define MCP_PAPER_WIDTH_A5_4000A		2528    // Add for ASK-4000A 2007/06/29 addition Ver.2.5.0

     /* For PaperSize.y parameter    */
    #define MCP_PAPER_HEIGHT_L              1074
    #define MCP_PAPER_HEIGHT_K              1228
    #define MCP_PAPER_HEIGHT_2L             2130
    #define MCP_PAPER_HEIGHT_6X8            2432
    #define MCP_PAPER_HEIGHT_A5             2748	// For ASK-2000
    #define MCP_PAPER_HEIGHT_8X10           3044    /* Add 2005/07/12 Ver.1.4.3 */
    #define MCP_PAPER_HEIGHT_8X12           3644    /* Add 2005/12/20 Ver.2.0.1 */
    #define MCP_PAPER_HEIGHT_8X4			1244    // Add for ASK-4000 2007/06/29 addition Ver.2.5.0
    #define MCP_PAPER_HEIGHT_A4				3556    // Add for ASK-4000A 2007/06/29 addition Ver.2.5.0
    #define MCP_PAPER_HEIGHT_A5_4000A		1800    // Add for ASK-4000A 2007/06/29 addition Ver.2.5.0

/* Image information    */
typedef struct {
    POINT       ImageData;      /* Image size ( x : width [pixel], y : height [pixel] )                                   */
                                /* (Indicate same size to paper size.)                                                    */
    POINT       Offset;         /* Output offset ( x : On paper width direction, y : On paper height direction [pixel] )  */
                                /* (* No use for this SDK)                                                                */
} MCP_IMAGEINFO, *PMCP_IMAGEINFO;

    /* Paper direction of x, y coordinates                  */
    /*                                                      */
    /*  x : Printer head main-scanning direction            */
    /*  y : Printer head sub-scanning direction             */
    /*                                                      */
    /*      0  ---->  x                                     */
    /*        +---------+  +---------+                      */
    /*      | |         |  |         |                      */
    /*      v | L size  |  |         |                      */
    /*        | paper   |  | 2L size |                      */
    /*      y +---------+  | paper   |                      */
    /*                     |         |                      */
    /*                     |         |                      */
    /*                     +---------+                      */
    /*                                                      */


/* Color correct information    */
typedef struct {
    short       nSharpness;     /* Printer sharpness correct    */
    int         nReserved;      /* (* No use for this SDK)      */
} MCP_COLORINFO, *PMCP_COLORINFO;

    /* For parameter nSharpness value   */
    #define MCP_SHARPNESS_MAX               (7)
    #define MCP_SHARPNESS_MIN               (0)
    #define MCP_SHARPNESS_DEFAULT           (0)   /* Changed 2005/06/22 Ver.1.3.1 */
    #define MCP_SHARPNESS_DEFAULT_1500      (0)   /* 2006/03/30 Ver.2.3.0 */
    #define MCP_SHARPNESS_DEFAULT_4000      (2)   /* 2006/03/30 Ver.2.3.0 */
    #define MCP_SHARPNESS_DEFAULT_2000      (4)   /* 2007/09/26 Ver.2.6.1 */
//  #define MCP_SHARPNESS_DEFAULT_2000      (1)   /* 2006/03/30 Ver.2.3.0 */
    #define MCP_SHARPNESS_DEFAULT_4000A     (2)   /* 2007/06/29 Ver.2.5.0 */
    #define MCP_SHARPNESS_DEFAULT_2500      (4)   /* 2008/10/09 Ver.2.6.8 */
    #define MCP_SHARPNESS_NOCORRECT         (-1)  /* Not execute sharpness correct  */

/* Print information    */
typedef struct {
    WORD            wCopies;            /* Copies count                         */
    MCP_PAPERINFO   PaperInfo;          /* Paper information                    */
    MCP_IMAGEINFO   ImageInfo;          /* Image information                    */
    MCP_COLORINFO   ColorInfo;          /* Color correct information            */
    short           nPrintMode;         /* Print mode (for ASK-1500 only)       */
} MCP_PRINTINFO, *PMCP_PRINTINFO;

    /* For parameter wCopies max value  */
    #define MCP_MAX_COPIES                  1000

    /* For parameter nPrintMode value ( Only ASK-1500 )  */       
    #define MCP_PRINT_MODE_SPEED            (0)  /* Print speed priority mode   */
    #define MCP_PRINT_MODE_QUALITY          (1)  /* Print quality priority mode */


/* Printer printing information    */
typedef struct {
    BYTE        bDetailStatus;          /* Printer status                       */
    WORD        wJobNumber;             /* Print job number (1, 2, 3, ...)      */
    WORD        wPrintedNumber;         /* Printed copies                       */
    WORD        wPrintRequestNumber;    /* Requested copies                     */
} MCP_PRINTINGINFO, *PMCP_PRINTINGINFO;

    /* For bDetailStatus parameter     */
    #define MCP_PRINTERSTATUS_READY          0x00
    #define MCP_PRINTERSTATUS_PRINTING       0x01
    #define MCP_PRINTERSTATUS_INITIALIZING   0x02
    #define MCP_PRINTERSTATUS_FEEDANDCUT     0x03    /* (* No use for this SDK)  */
    #define MCP_PRINTERSTATUS_JOB_CANCEL     0xF0    /* Canceled print on the way of multi copies is end.    */  /* Add 2005/04/19 Ver.1.1.0 */
    #define MCP_PRINTERSTATUS_SEND_COMPLETE  0xF1
    #define MCP_PRINTERSTATUS_START_SENDING  0xF2    /* Image data transfer start after image process ended in PC.   */  /* Add 2005/05/11 Ver.1.2.0 */
    #define MCP_PRINTERSTATUS_START_RECOVERY 0xF3    /* Printer star recovery after recoverable error notify.   */  /* Add 2005/06/08 Ver.1.3.0 */


/* Printer software version information     */
typedef struct {
    char        szVersion1[64];         /* Main firmware version    */
    char        szVersion2[64];         /* (* No use for this SDK)  */
    char        szVersion3[64];         /* (* No use for this SDK)  */
} MCP_VERSIONINFO, *PMCP_VERSIONINFO;



/************************************************/
/*                                              */
/*  IniFile definition                          */
/*                                              */
/************************************************/

/* IniFile Name  */
#define MCP_INIFILE_NAME	"CCORRECT.INI"	/* 2005/06/24 Ver.1.4.0 */



/************************************************/
/*                                              */
/*  Export function prototype                   */
/*                                              */
/************************************************/


/********************************************************************************/
/* FunctionName : MCP_Init                                                      */
/* FunctionType : DWORD MCP_Init(void);                                         */
/* Arguments    :                                                               */
/* ReturnValue  : DWORD;                                                        */
/*                      MCP_OK : OK,  Else : Error info.                        */
/* Description  : Initialize SDK printer control.                               */
/* Notes        :                                                               */
/*                                      CREATE      2005/03/17  fujiwara        */
/********************************************************************************/
DWORD WINAPI MCP_Init(void);


/********************************************************************************/
/* FunctionName : MCP_Finish                                                    */
/* FunctionType : DWORD MCP_Finish(void);                                       */
/* Arguments    :                                                               */
/* ReturnValue  : DWORD;                                                        */
/*                      MCP_OK : OK,  Else : Error info.                        */
/* Description  : Finish SDK printer control.                                   */
/* Notes        :                                                               */
/*                                      CREATE      2005/03/17  fujiwara        */
/********************************************************************************/
DWORD WINAPI MCP_Finish(void);


/********************************************************************************/
/* FunctionName : MCP_EnumPrinter                                               */
/* FunctionType : DWORD MCP_EnumPrinter(PUINT puiNumPrinter, PHANDLE phPrinter);*/
/* Arguments    : puiNumPrinter [IN]  Number of printer handles want to get.    */
/*                              [OUT] Number of found printer handles.          */
/*                phPrinter     [OUT] Printer handles array.                    */
/* ReturnValue  : DWORD;                                                        */
/*                      MCP_OK : OK,  Else : Error info.                        */
/* Description  : Search target printers.                                       */
/* Notes        :                                                               */
/*                                      CREATE      2005/03/17  fujiwara        */
/********************************************************************************/
DWORD WINAPI MCP_EnumPrinter(PUINT puiNumPrinter, PHANDLE phPrinter);


/********************************************************************************/
/* FunctionName : MCP_OpenPrinter                                               */
/* FunctionType : DWORD MCP_OpenPrinter(HANDLE hPrinter, DWORD dwIdleTimeout);  */
/* Arguments    : hPrinter      [IN]  Handle of printer.                        */
/*                dwIdleTimeout [IN]  Idle timeout setting. (Second)            */
/*                                    (Set MCP_TIMEOUT_INFINITE.)               */
/* ReturnValue  : DWORD;                                                        */
/*                      MCP_OK : OK,  Else : Error info.                        */
/* Description  : Start control specified printer. (Connect printer)            */
/* Notes        :                                                               */
/*                                      CREATE      2005/03/17  fujiwara        */
/********************************************************************************/
DWORD WINAPI MCP_OpenPrinter(HANDLE hPrinter, DWORD dwIdleTimeout);

/* For parameter dwIdleTimeout max value    */
#define MCP_TIMEOUT_INFINITE                0       /* Not set idle time-out    */


/********************************************************************************/
/* FunctionName : MCP_ClosePrinter                                              */
/* FunctionType : DWORD MCP_ClosePrinter(HANDLE hPrinter);                      */
/* Arguments    : hPrinter      [IN]  Handle of printer.                        */
/* ReturnValue  : DWORD;                                                        */
/*                      MCP_OK : OK,  Else : Error info.                        */
/* Description  : Close control specified printer. (Disconnect printer)         */
/* Notes        :                                                               */
/*                                      CREATE      2005/03/17  fujiwara        */
/********************************************************************************/
DWORD WINAPI MCP_ClosePrinter(HANDLE hPrinter);


/********************************************************************************/
/* FunctionName : MCP_GetPrinterInfo                                            */
/* FunctionType : DWORD MCP_GetPrinterInfo(HANDLE hPrinter,                     */
/*                      PMCP_PRINTER_INFO pPrinterInfo);                        */
/* Arguments    : hPrinter      [IN]  Handle of printer.                        */
/*                pPrinterInfo  [OUT] Printer device information.               */
/* ReturnValue  : DWORD;                                                        */
/*                      MCP_OK : OK,  Else : Error info.                        */
/* Description  : Get printer device information.                               */
/* Notes        :                                                               */
/*                                      CREATE      2005/03/17  fujiwara        */
/********************************************************************************/
DWORD WINAPI MCP_GetPrinterInfo(HANDLE hPrinter, PMCP_PRINTER_INFO pPrinterInfo);


/********************************************************************************/
/* FunctionName : MCP_InitPrintSetting                                          */
/* FunctionType : DWORD MCP_InitPrintSetting(HANDLE hPrinter);                  */
/* Arguments    : hPrinter      [IN]  Handle of printer.                        */
/* ReturnValue  : DWORD;                                                        */
/*                      MCP_OK : OK,  Else : Error info.                        */
/* Description  : Initialize parameters of SDK print setting area with default  */
/*                value.                                                        */
/* Notes        :                                                               */
/*                                      CREATE      2005/03/17  fujiwara        */
/********************************************************************************/
DWORD WINAPI MCP_InitPrintSetting(HANDLE hPrinter);


/********************************************************************************/
/* FunctionName : MCP_SendPrintSetting                                          */
/* FunctionType : DWORD MCP_SendPrintSetting(HANDLE hPrinter,                   */
/*                      MCP_PRINTINFO PrintInfo);                               */
/* Arguments    : hPrinter          [IN]  Handle of printer.                    */
/*                PrintInfo         [OUT] Print setting information.            */
/* ReturnValue  : DWORD;                                                        */
/*                      MCP_OK : OK,  Else : Error info.                        */
/* Description  : Send print setting to the target printer.                     */
/* Notes        :                                                               */
/*                                      CREATE      2005/03/17  fujiwara        */
/********************************************************************************/
DWORD WINAPI MCP_SendPrintSetting(HANDLE hPrinter, MCP_PRINTINFO PrintInfo);


/********************************************************************************/
/* FunctionName : MCP_InitPrinterMemory                                         */
/* FunctionType : DWORD MCP_InitPrinterMemory(HANDLE hPrinter);                 */
/* Arguments    : hPrinter          [IN]  Handle of printer.                    */
/* ReturnValue  : DWORD;                                                        */
/*                      MCP_OK : OK,  Else : Error info.                        */
/* Description  : Clear printer frame memory.                                   */
/* Notes        : Do not anything in this SDK.                                  */
/*                                      CREATE      2005/03/17  fujiwara        */
/********************************************************************************/
DWORD WINAPI MCP_InitPrinterMemory(HANDLE hPrinter);


/********************************************************************************/
/* FunctionName : (User program)                                                */
/* FunctionType : void CALLBACK callbackProc(HANDLE hPrinter,                   */
/*                      PMCP_PRINTINGINFO pPrintingInfo,                        */
/*                      DWORD dwErrorInfo);                                     */
/* Arguments    : hPrinter      [IN]  Handle of printer.                        */
/*                pPrintingInfo [IN]  Printing progress information.            */
/*                dwErrorInfo   [IN]  Error information.                        */
/*                              (Same type to SDK return code.)                 */
/* ReturnValue  : DWORD;                                                        */
/*                      MCP_OK : OK,  Else : Error info.                        */
/* Description  : Callback function resisted and called from                    */
/*                MCP_SendImageAndPrint() to notice print progress.             */
/* Notes        : The values pointed by pPrintingInfo are valid in only this    */
/*                function.                                                     */
/*                You should return this callback function to SDK quickly       */
/*                for not stop movement of SDK thread for printing.             */
/*                                                                              */
/********************************************************************************/
/* Pointer type "PMCP_CALLBACK" for this callback function   */
typedef void  (CALLBACK *PMCP_CALLBACK)(HANDLE, PMCP_PRINTINGINFO, DWORD);


/* Max count of thread that can start following functions for printing.     */
#define MCP_PRINTTHREAD_MAX_COUNT           2


/********************************************************************************/
/* FunctionName : MCP_SendImageAndPrint                                         */
/* FunctionType : DWORD MCP_SendImageAndPrint(HANDLE hPrinter, HANDLE hDib,     */
/*                      PMCP_CALLBACK pCallbackProc);                           */
/* Arguments    : hPrinter      [IN]  Handle of printer.                        */
/*                hDib          [IN]  Memory handle of print 24 Bpp DIB data    */
/*                                    after BITMAPINFO.                         */
/*                pCallbackProc [IN]  Address of user program function report   */
/*                                    to sending and printing progress.         */
/* ReturnValue  : DWORD;                                                        */
/*                      MCP_OK : OK,  Else : Error info.                        */
/* Description  : Send the image data to printer, start print, checking print   */
/*                end, and report sending and printing progress via             */
/*                pCallbackProc function.                                       */
/* Notes        : This function makes a thread to print.                        */
/*                So this function returns quickly.                             */
/*                                      CREATE      2005/03/17  fujiwara        */
/********************************************************************************/
DWORD WINAPI MCP_SendImageAndPrint(HANDLE hPrinter, HANDLE hDib, PMCP_CALLBACK pCallbackProc);


/********************************************************************************/
/* FunctionName : MCP_ReceivePrintedInfo                                        */
/* FunctionType : DWORD MCP_ReceivePrintedInfo(HANDLE hPrinter,                 */
/*                          PDWORD pdwTotalPrintCount);                         */
/* Arguments    : hPrinter           [IN]  Handle of printer.                   */
/*                pdwTotalPrintCount [OUT] Printer cumulative printed count     */
/*                                         after shipment.                      */
/* ReturnValue  : DWORD;                                                        */
/*                      MCP_OK : OK,  Else : Error info.                        */
/* Description  : Receive printer cumulative printed info.                      */
/* Notes        :                                                               */
/*                                      CREATE      2005/04/04  fujiwara        */
/********************************************************************************/
DWORD WINAPI MCP_ReceivePrintedInfo(HANDLE hPrinter, PDWORD pdwTotalPrintCount);


/********************************************************************************/
/* FunctionName : MCP_ReceiveHeadPrintedInfo			                        */
/* FunctionType : DWORD MCP_ReceiveHeadPrintedInfo(HANDLE hPrinter,				*/
/*                          PDWORD pdwTotalPrintCount);                         */
/* Arguments    : hPrinter           [IN]  Handle of printer.                   */
/*                pdwTotalPrintCount [OUT] Printer cumulative printed count     */
/*                                         after head change.                   */
/* ReturnValue  : DWORD;                                                        */
/*                      MCP_OK : OK,  Else : Error info.                        */
/* Description  : Receive printer cumulative printed info.                      */
/* Notes        :                                                               */
/*                                      CREATE      2008/02/10  nakashima       */
/********************************************************************************/
DWORD WINAPI MCP_ReceiveHeadPrintedInfo(HANDLE hPrinter, PDWORD pdwTotalPrintCount);


/********************************************************************************/
/* FunctionName : MCP_ReceiveVerInfo                                            */
/* FunctionType : DWORD MCP_ReceiveVerInfo(HANDLE hPrinter,                     */
/*                          PMCP_VERSIONINFO pVersionInfo);                     */
/* Arguments    : hPrinter      [IN]  Handle of printer.                        */
/*                pVersionInfo  [OUT] Printer software version info.            */
/* ReturnValue  : DWORD;                                                        */
/*                      MCP_OK : OK,  Else : Error info.                        */
/* Description  : Receive printer software version info.                        */
/* Notes        :                                                               */
/*                                      CREATE      2005/04/04  fujiwara        */
/********************************************************************************/
DWORD WINAPI MCP_ReceiveVerInfo(HANDLE hPrinter, PMCP_VERSIONINFO pVersionInfo);


/********************************************************************************/
/* FunctionName : MCP_GetPrinterStatus                                          */
/* FunctionType : DWORD MCP_GetPrinterStatus(HANDLE hPrinter,                   */
/*                      PDWORD pdwPrinterStatus);                               */
/* Arguments    : hPrinter          [IN]  Handle of printer.                    */
/*                pdwPrinterStatus  [OUT] Printer status.                       */
/*                          (same to MCP_PRINTINGINFO.bDetailStatus value.      */
/*                           MCP_PRINTERSTATUS_XXX)                             */
/* ReturnValue  : DWORD;                                                        */
/*                      MCP_OK : OK,  Else : Error info.                        */
/* Description  : Receive printer status.                                       */
/* Notes        :                                                               */
/*                                      CREATE      2005/03/17  fujiwara        */
/********************************************************************************/
DWORD WINAPI MCP_GetPrinterStatus(HANDLE hPrinter, PDWORD pdwPrinterStatus);


/********************************************************************************/
/* FunctionName : MCP_GetPaperRemain                                            */
/* FunctionType : DWORD MCP_GetPaperRemain(HANDLE hPrinter,                     */
/*                          PDWORD pdwPaperRemain);                             */
/* Arguments    : hPrinter          [IN]  Handle of printer.                    */
/*                pdwPaperRemain    [OUT] Printer paper remain copies.          */
/* ReturnValue  : DWORD;                                                        */
/*                      MCP_OK : OK,  Else : Error info.                        */
/* Description  : Receive printer paper remain copies.                          */
/* Notes        :                                                               */
/*                                      CREATE      2005/04/04  fujiwara        */
/********************************************************************************/
DWORD WINAPI MCP_GetPaperRemain(HANDLE hPrinter, PDWORD pdwPaperRemain);

/* For parameter pdwPaperRemain value    */
#define MCP_PAPER_REMAIN_UNKNOWN            0xFFFFFFFF  /* Printer not printed after turn on.   */


/********************************************************************************/
/* FunctionName : MCP_GetLastError                                              */
/* FunctionType : DWORD MCP_GetLastError(HANDLE hPrinter);                      */
/* Arguments    : hPrinter      [IN]  Handle of printer.                        */
/* ReturnValue  : DWORD;                                                        */
/*                      Error code.                                             */
/* Description  : Get SDK last error code.                                      */
/* Notes        : if hPrinter indicated NULL, ReturnValue is error code of      */
/*                SDK all.                                                      */
/*                                      CREATE      2005/03/17  fujiwara        */
/********************************************************************************/
DWORD WINAPI MCP_GetLastError(HANDLE hPrinter);


/********************************************************************************/
/* FunctionName : MCP_CancelPrint                                               */
/* FunctionType : DWORD MCP_CancelPrint(HANDLE hPrinter);                       */
/* Arguments    : hPrinter      [IN]  Handle of printer.                        */
/* ReturnValue  : DWORD;                                                        */
/*                      MCP_OK : OK,  Else : Error info.                        */
/* Description  : Cancel printing multi copies in the way.                      */
/* Notes        :                                                               */
/*                                      CREATE      2005/03/17  fujiwara        */
/********************************************************************************/
DWORD WINAPI MCP_CancelPrint(HANDLE hPrinter);


/********************************************************************************/
/* FunctionName : MCP_StopPrint                                                 */
/* FunctionType : DWORD MCP_StopPrint(HANDLE hPrinter);                         */
/* Arguments    : hPrinter      [IN]  Handle of printer.                        */
/* ReturnValue  : DWORD;                                                        */
/*                      MCP_OK : OK,  Else : Error info.                        */
/* Description  : Stop printing and Clear spooled image data.                   */
/* Notes        : This function abort the printing progress watch.              */
/*                                      CREATE      2005/12/09  Mochida         */
/********************************************************************************/
DWORD WINAPI MCP_StopPrint(HANDLE hPrinter);// Ver.1.9.0


/********************************************************************************/
/* FunctionName : MCP_LutConversionForPrinter                                   */
/* FunctionType : DWORD MCP_LutConversionForPrinter(HANDLE hDib,                */
/*                          PCSTR strDataFileName);                             */
/* Arguments    : hDib          [IN]  Memory handle of image 24 Bpp DIB data    */
/*                                    after BITMAPINFO.                         */
/* Arguments    : strDataFileName [IN]  File name of 3D LUT parameter data      */
/*                                      file. (File name only (Not full-path.)) */
/* ReturnValue  : DWORD;                                                        */
/*                      MCP_OK : OK,  Else : Error info.                        */
/* Description  : Execute 3D LUT conversion to indicated image data.            */
/* Notes        :                                                               */
/*                                      CREATE      2005/01/13  fujiwara        */
/********************************************************************************/
DWORD WINAPI MCP_LutConversionForPrinter(HANDLE hDib, PCSTR strDataFileName);


/********************************************************************************/
/* FunctionName : MCP_SetPrinterMemoryClearFlag                                 */
/* FunctionType : DWORD MCP_SetPrinterMemoryClearFlag(HANDLE hPrinter);         */
/* Arguments    : hPrinter          [IN]  Handle of printer.                    */
/* ReturnValue  : DWORD;                                                        */
/*                      MCP_OK : OK,  Else : Error info.                        */
/* Description  : Set mode that clear printer's memory memory at executing next */
/*                MCP_SendPrintSetting().                                       */
/* Notes        :                                                               */
/*                                      CREATE      2005/02/08  fujiwara        */
/********************************************************************************/
DWORD WINAPI MCP_SetPrinterMemoryClearFlag(HANDLE hPrinter);
/* (* No use for this SDK)  */


/********************************************************************************/
/* FunctionName : MCP_Generate1dData                                            */
/* FunctionType : DWORD MCP_Generate1dData(LPBYTE lpbData,                      */
/*                          int nDdata, int nCdata, int nMdata, intYdata);      */
/* Arguments    : lpbData [OUT] Memory to hold 1d data.                         */
/*                nDdata  [IN]  D parameter.                                    */
/*                nCdata  [IN]  C parameter.                                    */
/*                nMdata  [IN]  M parameter.                                    */
/*                nYdata  [IN]  Y parameter.                                    */
/* ReturnValue  : DWORD;                                                        */
/*                      MCP_OK : OK,  Else : Error info.                        */
/* Description  : Generate 1d data to use DCMY parameter that Operater inputs.  */
/* Notes        : Allocate memory to hold 1d data by application.               */
/*                                      CREATE      2005/06/24  mochida         */
/********************************************************************************/
DWORD WINAPI MCP_Generate1dData(LPBYTE lpbData, int nDdata, int nCdata, int nMdata, int nYdata);    /* 2005/06/24 Ver.1.4.0 */


/********************************************************************************/
/* FunctionName : MCP_Generate1dDataWithGammaCurve                              */
/* FunctionType : DWORD MCP_Generate1dDataWithGammaCurve(LPBYTE lpbData,        */
/*                          int nDdata, int nCdata, int nMdata, intYdata);      */
/* Arguments    : lpbData [OUT] Memory to hold 1d data.                         */
/*                nDdata  [IN]  D parameter.                                    */
/*                nCdata  [IN]  C parameter.                                    */
/*                nMdata  [IN]  M parameter.                                    */
/*                nYdata  [IN]  Y parameter.                                    */
/* ReturnValue  : DWORD;                                                        */
/*                      MCP_OK : OK,  Else : Error info.                        */
/* Description  : Generate 1d data to use DCMY parameter that Operater inputs.  */
/*                0 and 255 are fixed. Other value increase or decrease.        */
/* Notes        : Allocate memory to hold 1d data by application.               */
/*                                      CREATE      2006/07/27  ksuga           */
/********************************************************************************/
DWORD WINAPI MCP_Generate1dDataWithGammaCurve(LPBYTE lpbData, int nDdata, int nCdata, int nMdata, int nYdata);    /* 2005/06/24 Ver.1.4.0 */

/********************************************************************************/
/* FunctionName : MCP_Make1dDataLinear				                            */
/* FunctionType : DWORD MCP_Make1dDataLinear(LPBYTE lpbData,				    */
/*                          int nRdata, int nGdata, intBdata);				    */
/* Arguments    : lpbData [OUT] Memory to hold 1d data.                         */
/*                nRdata  [IN]  R parameter.                                    */
/*                nGdata  [IN]  G parameter.                                    */
/*                nBdata  [IN]  B parameter.                                    */
/* ReturnValue  : DWORD;                                                        */
/*                      MCP_OK : OK,  Else : Error info.                        */
/* Description  : Make 1d data to use RGB parameter that Operater inputs.		*/
/*                255 is fixed. Other value increase or decrease.				*/
/* Notes        : Allocate memory to hold 1d data by application.               */
/*                                      CREATE      2007/08/09  nakashima       */
/********************************************************************************/
DWORD WINAPI MCP_Make1dDataLinear(LPBYTE lpbData, int nRdata, int nGdata, int nBdata);    /* 2005/06/24 Ver.1.4.0 */

/********************************************************************************/
/* FunctionName : MCP_Make1dDataGammaCurve			                            */
/* FunctionType : DWORD MCP_Make1dDataGammaCurve(LPBYTE lpbData,			    */
/*                          int nRdata, int nGdata, intBdata);				    */
/* Arguments    : lpbData [OUT] Memory to hold 1d data.                         */
/*                nRdata  [IN]  R parameter.                                    */
/*                nGdata  [IN]  G parameter.                                    */
/*                nBdata  [IN]  B parameter.                                    */
/* ReturnValue  : DWORD;                                                        */
/*                      MCP_OK : OK,  Else : Error info.                        */
/* Description  : Make 1d data to use RGB parameter that Operater inputs.		*/
/*                0 and 255 are fixed. Other value increase or decrease.        */
/* Notes        : Allocate memory to hold 1d data by application.               */
/*                                      CREATE      2007/08/09  nakashima       */
/********************************************************************************/
DWORD WINAPI MCP_Make1dDataGammaCurve(LPBYTE lpbData, int nRdata, int nGdata, int nBdata);    /* 2005/06/24 Ver.1.4.0 */


/********************************************************************************/
/* FunctionName : MCP_Create1dDataFile                                          */
/* FunctionType : DWORD MCP_Create1dDataFile(HANDLE hPrinter                    */
/*                          LPBYTE lpbData, char* szFileName);                  */
/* Arguments    : hPrinter      [IN]  Handle of printer.                        */
/*                lpbData       [OUT] Memory to hold 1d data.                   */
/*                szFileName    [IN]  Directory to save 1d data file.           */
/*                              [OUT] 1d data file name.(Full path)             */
/* ReturnValue  : DWORD;                                                        */
/*                      MCP_OK : OK,  Else : Error info.                        */
/* Description  : Create 1d data File that corresponds a printer.               */
/* Notes        : Put 1d data into lpbData to call MCP_Generate1dData()         */
/*                before this function.                                         */
/*                                      CREATE      2005/06/24  mochida         */
/********************************************************************************/
DWORD WINAPI MCP_Create1dDataFile(HANDLE hPrinter, LPBYTE lpbData, char* szFileName);   /* 2005/06/24 Ver.1.4.0 */


/********************************************************************************/
/* FunctionName : MCP_Delete1dDataFile                                          */
/* FunctionType : DWORD MCP_Delete1dDataFile(HANDLE hPrinter, char* szFileName);*/
/* Arguments    : hPrinter      [IN]  Handle of printer.                        */
/*                szFileName    [IN]  Directory where 1d data file is saved.    */
/*                              [OUT] Deleted 1d data file name.(Full path)     */
/* ReturnValue  : DWORD;                                                        */
/*                      MCP_OK : OK,  Else : Error info.                        */
/* Description  : Delete 1d data File that corresponds a printer.               */
/*                                      CREATE      2005/06/24  mochida         */
/********************************************************************************/
DWORD WINAPI MCP_Delete1dDataFile(HANDLE hPrinter, char* szFileName);   /* 2005/06/24 Ver.1.4.0 */


/********************************************************************************/
/* FunctionName : MCP_Search1dDataFile                                          */
/* FunctionType : DWORD MCP_Search1dDataFile(HANDLE hPrinter, char* szFileName);*/
/* Arguments    : hPrinter      [IN]  Handle of printer.                        */
/*                szFileName    [IN]  Directory where 1d data file is saved.    */
/*                              [OUT] Found 1d data file name.(Full path)       */
/* ReturnValue  : DWORD;                                                        */
/*                      MCP_OK : OK,  Else : Error info.                        */
/* Description  : Search 1D Data File that corresponds a printer.               */
/* Notes        : When 1D data file isn't found, NULL is assigned as szFileName.*/
/*                                      CREATE      2005/06/24  mochida         */
/********************************************************************************/
DWORD WINAPI MCP_Search1dDataFile(HANDLE hPrinter, char* szFileName);   /* 2005/06/24 Ver.1.4.0 */


/********************************************************************************/
/* FunctionName : MCP_Do1dLut                                                   */
/* FunctionType : DWORD MCP_Do1dLut(HANDLE hDib, char* szFileName);             */
/* Arguments    : hDib          [IN]  DIB handle for image data.                */
/*                szFileName    [IN]  1d data file name.(Full path)             */
/* ReturnValue  : DWORD;                                                        */
/*                      MCP_OK : OK,  Else : Error info.                        */
/* Description  : Execute 1D LUT conversion to the indicated image data.        */
/* Notes        : Set the memory handle after BITMAPINFOHEADER of DIB to hDib.  */
/*                This function can accept only DIB of 24bit.                   */
/*                                      CREATE      2005/06/24  mochida         */
/********************************************************************************/
DWORD WINAPI MCP_Do1dLut(HANDLE hDib, char* szFileName);    /* 2005/06/24 Ver.1.4.0 */


/********************************************************************************/
/* FunctionName : MCP_Exec1dConversion                                          */
/* FunctionType : DWORD MCP_Exec1dConversion(HANDLE hPrinter, HANDLE hDib);     */
/* Arguments    : hPrinter      [IN]  Handle of printer.                        */
/*                hDib          [IN]  DIB handle for image data.                */
/* ReturnValue  : DWORD;                                                        */
/*                      MCP_OK : OK,  Else : Error info.                        */
/* Description  : Execute 1D LUT conversion to the indicated image data.        */
/* Notes        : Application calls this function to perform 1d LUT conversion. */
/*                Set the memory handle after BITMAPINFOHEADER of DIB to hDib.  */
/*                This function can accept only DIB of 24bit.                   */
/*                In this function, it is called MCP_Search1dDataFile() and     */
/*                MCP_Do1dLut().                                                */
/*                                      CREATE      2005/06/24  mochida         */
/********************************************************************************/
DWORD WINAPI MCP_Exec1dConversion(HANDLE hPrinter, HANDLE hDib);    /* 2005/06/24 Ver.1.4.0 */


/********************************************************************************/
/* FunctionName : MCP_SendImageAndMultiPrint                                    */
/* FunctionType : DWORD MCP_SendImageAndPrint(HANDLE hPrinter, HANDLE hDib1st,  */
/*                      HANDLE hDib2nd, PMCP_CALLBACK pCallbackProc);           */
/* Arguments    : hPrinter      [IN]  Handle of printer.                        */
/*                hDib1st       [IN]  24 Bpp DIB data for 1st image.            */
/*                hDib2nd       [IN]  24 Bpp DIB data for 2nd image.            */
/*                pCallbackProc [IN]  Address of user program function report   */
/*                                    to sending and printing progress.         */
/* ReturnValue  : DWORD;                                                        */
/*                      MCP_OK : OK,  Else : Error info.                        */
/* Description  : Send the image data to printer, start print, checking print   */
/*                end, and report sending and printing progress via             */
/*                pCallbackProc function.                                       */
/* Notes        : This function do "Multi cut print".(The print processing of   */
/*                as much as two K sizes is done to 2K size form.)              */
/*                If 1 image print, NULL is set in hDib2nd.                     */
/*                                      CREATE      2005/07/04  mochida         */
/********************************************************************************/
DWORD WINAPI MCP_SendImageAndMultiPrint(HANDLE hPrinter, HANDLE hDib1st, HANDLE hDib2nd, PMCP_CALLBACK pCallbackProc);/* 2005/07/21 Ver.1.4.4 */

/********************************************************************************/
/* FunctionName : MCP_SendImageAndThreeDivPrint                                 */
/* FunctionType : DWORD MCP_SendImageAndPrint(HANDLE hPrinter, HANDLE hDib1st,  */
/*                      HANDLE hDib2nd, HANDLE hDib3rd,							*/
/*						PMCP_CALLBACK pCallbackProc);							*/
/* Arguments    : hPrinter      [IN]  Handle of printer.                        */
/*                hDib1st       [IN]  24 Bpp DIB data for 1st image.            */
/*                hDib2nd       [IN]  24 Bpp DIB data for 2nd image.            */
/*                hDib3rd       [IN]  24 Bpp DIB data for 3rd image.            */
/*                pCallbackProc [IN]  Address of user program function report   */
/*                                    to sending and printing progress.         */
/* ReturnValue  : DWORD;                                                        */
/*                      MCP_OK : OK,  Else : Error info.                        */
/* Description  : Send the image data to printer, start print, checking print   */
/*                end, and report sending and printing progress via             */
/*                pCallbackProc function.                                       */
/* Notes        : This function do "Multi cut print".(The print processing of   */
/*                as much as three 8*4 sizes is done to 8*12 size form.)        */
/*                If 1 image print, NULL is set in hDib2nd and hDib3rd.         */
/*                If 2 images print, NULL is set in hDib3rd.			        */
/*                                      CREATE      2007/06/26  nakashima       */
/********************************************************************************/
DWORD WINAPI MCP_SendImageAndThreeDivPrint(HANDLE hPrinter, HANDLE hDib1st, HANDLE hDib2nd, HANDLE hDib3rd, PMCP_CALLBACK pCallbackProc); /* 2007/06/29 Ver.2.5.0 */

/********************************************************************************/
/* FunctionName : MCP_SetUsbNo                                                  */
/* FunctionType : DWORD MCP_SetUsbNo(HANDLE hPrinter, int iNewUsbNo);           */
/* Arguments    : hPrinter      [IN]  Handle of printer.                        */
/*                iNewUsbNo     [IN]  New Usb Number.                           */
/* ReturnValue  : DWORD;                                                        */
/*                      MCP_OK : OK,  Else : Error info.                        */
/* Description  : When Usb No. overlaps, Set New Usb No. with this function.    */
/* Notes        :                                                               */
/*                                      CREATE      2005/12/08  mochida         */
/********************************************************************************/
DWORD WINAPI MCP_SetUsbNo(HANDLE hPrinter, int iNewUsbNo);/* 2005/12/08 Ver.1.8.0 */

/********************************************************************************/
/* FunctionName : MCP_SetAutoRecoveryFlag                                       */
/* FunctionType : DWORD MCP_SetAutoRecoveryFlag(HANDLE hPrinter, BOOL fAuto);   */
/* Arguments    : hPrinter      [IN]  Handle of printer.                        */
/*                fAuto         [IN]  Auto Recovery Flag.                       */
/* ReturnValue  : DWORD;                                                        */
/*                      MCP_OK : OK,  Else : Error info.                        */
/* Description  : Set flag that wheather the print process continue  or abort,  */
/*                when Recoverable Error occurs.                                */
/* Notes        :                                                               */
/*                                      CREATE      2005/12/08  mochida         */
/********************************************************************************/
DWORD WINAPI MCP_SetAutoRecoveryFlag(HANDLE hPrinter, BOOL fAuto);/* 2005/12/08 Ver.1.8.0 */

/********************************************************************************/
/* FunctionName : MCP_GetPrinterError                                           */
/* FunctionType : DWORD MCP_GetPrinterError(HANDLE hPrinter,                    */
/*                                          char*  szPrinterError,              */
/*                                          DWORD* dwPrinterErrorDetail);       */
/* Arguments    : hPrinter       [IN]  Handle of printer.                       */
/*                szPrinterError       [OUT]  Printer error code.(string type)  */
/*                dwPrinterErrorDetail [OUT]  Status of printer error.          */
/* ReturnValue  : DWORD;                                                        */
/*                      MCP_OK : OK,  Else : Error info.                        */
/* Description  : This function is called after when there is an error response */
/*                in API or the callback function.                              */
/*                When it is not a printer error, NULL is returned to           */
/*                szPrinterError.                                               */
/* Notes        : ASK-1500 return NULL to szPrinterError,                       */
/*                but dwPrinterErrorDetail is available.                        */
/*                                                                              */
/*                                      CREATE      2006/03/13  mochida         */
/********************************************************************************/
DWORD WINAPI MCP_GetPrinterError(HANDLE hPrinter, char* szPrinterError, DWORD* dwPrinterErrorDetail);/* 2006/03/13 Ver.2.2.0 */


/* Pointer type for SDK export functions    */

typedef DWORD (WINAPI *PMCP_Init)(void);
typedef DWORD (WINAPI *PMCP_Finish)(void);
typedef DWORD (WINAPI *PMCP_EnumPrinter)(PUINT, PHANDLE);
typedef DWORD (WINAPI *PMCP_OpenPrinter)(HANDLE, DWORD);
typedef DWORD (WINAPI *PMCP_ClosePrinter)(HANDLE);
typedef DWORD (WINAPI *PMCP_GetPrinterInfo)(HANDLE, PMCP_PRINTER_INFO);
typedef DWORD (WINAPI *PMCP_InitPrintSetting)(HANDLE);
typedef DWORD (WINAPI *PMCP_SendPrintSetting)(HANDLE, MCP_PRINTINFO);
typedef DWORD (WINAPI *PMCP_InitPrinterMemory)(HANDLE);
typedef DWORD (WINAPI *PMCP_SendImageAndPrint)(HANDLE, HANDLE, PMCP_CALLBACK);
typedef DWORD (WINAPI *PMCP_ReceivePrintedInfo)(HANDLE, PDWORD);
typedef DWORD (WINAPI *PMCP_ReceiveHeadPrintedInfo)(HANDLE, PDWORD);
typedef DWORD (WINAPI *PMCP_ReceiveVerInfo)(HANDLE, PMCP_VERSIONINFO);
typedef DWORD (WINAPI *PMCP_GetPrinterStatus)(HANDLE, PDWORD);
typedef DWORD (WINAPI *PMCP_GetPaperRemain)(HANDLE, PDWORD);
typedef DWORD (WINAPI *PMCP_GetLastError)(HANDLE);
typedef DWORD (WINAPI *PMCP_CancelPrint)(HANDLE);
typedef DWORD (WINAPI *PMCP_LutConversionForPrinter)(HANDLE, PCSTR);
typedef DWORD (WINAPI *PMCP_SetPrinterMemoryClearFlag)(HANDLE);     /* (* No use for this SDK)  */
typedef DWORD (WINAPI *PMCP_Generate1dData)(LPBYTE, int, int, int, int);/* 2005/06/24 Ver.1.4.0 */
typedef DWORD (WINAPI *PMCP_Create1dDataFile)(HANDLE, LPBYTE, char*);   /* 2005/06/24 Ver.1.4.0 */
typedef DWORD (WINAPI *PMCP_Delete1dDataFile)(HANDLE, char*);           /* 2005/06/24 Ver.1.4.0 */
typedef DWORD (WINAPI *PMCP_Search1dDataFile)(HANDLE, char*);           /* 2005/06/24 Ver.1.4.0 */
typedef DWORD (WINAPI *PMCP_Do1dLut)(HANDLE, char*);                    /* 2005/06/24 Ver.1.4.0 */
typedef DWORD (WINAPI *PMCP_Exec1dConversion)(HANDLE, HANDLE);          /* 2005/06/24 Ver.1.4.0 */
typedef DWORD (WINAPI *PMCP_SendImageAndMultiPrint)(HANDLE, HANDLE, HANDLE, PMCP_CALLBACK);/* 2005/07/04 Ver.1.4.1 */
typedef DWORD (WINAPI *PMCP_SendImageAndThreeDivPrint)(HANDLE, HANDLE, HANDLE, HANDLE, PMCP_CALLBACK);/* 2007/06/29 Ver.2.5.0 */
typedef DWORD (WINAPI *PMCP_SetUsbNo)(HANDLE, int);                     /* 2005/12/08 Ver.1.8.0 */
typedef DWORD (WINAPI *PMCP_SetAutoRecoveryFlag)(HANDLE, BOOL);         /* 2005/12/08 Ver.1.8.0 */
typedef DWORD (WINAPI *PMCP_StopPrint)(HANDLE);                         /* 2005/12/09 Ver.1.9.0 */
typedef DWORD (WINAPI *PMCP_GetPrinterError)(HANDLE, char*, PDWORD);    /* 2006/03/13 Ver.2.2.0 */
typedef DWORD (WINAPI *PMCP_Generate1dDataWithGammaCurve)(LPBYTE, int, int, int, int);/* 2006/08/14 Ver.2.4.0 */
typedef DWORD (WINAPI *PMCP_Make1dDataLinear)(LPBYTE, int, int, int);	/* 2007/08/09 Ver.2.6.0 */
typedef DWORD (WINAPI *PMCP_Make1dDataGammaCurve)(LPBYTE, int, int, int);	/* 2007/08/09 Ver.2.6.0 */

/************************************************/
/*                                              */
/*  Error codes                                 */
/*                                              */
/************************************************/


#define MCP_OK                          0x00000000


/*
 *  Export functions return error information
 */

    /* Export functions return code structure                                   */
    /*                                                                          */
    /*       3 3 2 2 2 2 2 2 2 2 2 2 1 1 1 1 1 1 1 1 1 1                        */
    /*       1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0    */
    /*      +---+-+---------+---------------+---------------+---------------+   */
    /*      | 0 |C|  Flag   |  Error Info.  |   Recovery    | Reserved (=0) |   */
    /*      +---+-+---------+---------------+---------------+---------------+   */
    /*                                                                          */
    /*          C          : Customer error flag : 1                            */
    /*          Flag       : Error Flag                                         */
    /*          Recovery   : Recovery information                               */

/* Error Flag       */

#define MCP_ERRF_OK                     0x00    /* OK                                 */  
#define MCP_ERRF_P                      0x01    /* Printer condition error            */
#define MCP_ERRF_H                      0x02    /* Printer hardware error             */
#define MCP_ERRF_I                      0x03    /* Communication interface error      */
#define MCP_ERRF_S                      0x04    /* Software error                     */
#define MCP_ERRF_H_CONTINUE             0x12    /* Recoverable printer hardware error */    /* Add 2005/06/08 Ver.1.3.0 */


/* Macros for check whether error info is recoverable printer hardware error or not.  */  /* Add 2005/06/08 Ver.1.3.0 */

#define MCP_IS_ERRF_H_CONTINUE(dwRet)  ( (MCP_ERRFLAG(dwRet)==MCP_ERRF_H_CONTINUE) ? TRUE:FALSE ) 


/* Error Info       */

#define MCP_PERR_PRINTER_BUSY           0x01
#define MCP_PERR_NO_PRINTER             0x02
#define MCP_PERR_PRINTER_RESERVED       0x03
#define MCP_PERR_INVALID_CONNECT        0x04
#define MCP_PERR_UNKNOWN                0x3F

#define MCP_HERR_H                      0x01
#define MCP_HERR_J                      0x02
#define MCP_HERR_U                      0x03
#define MCP_HERR_C                      0x04
#define MCP_HERR_NO_PAPER               0x11
#define MCP_HERR_PARAMETER              0x12
#define MCP_HERR_UNKNOWN                0x3F

#define MCP_IERR_COMM_DEVICE            0x01
#define MCP_IERR_UNKNOWN                0x3F

#define MCP_SERR_USE_SDK                0x01
#define MCP_SERR_PC_RESOURCE            0x02
#define MCP_SERR_UNKNOWN                0x3F


/* Recovery Info    */

#define MCP_RECOVERY_IGNORE             0x01
#define MCP_RECOVERY_CMD_RETRY          0x02
#define MCP_RECOVERY_WAIT_READY         0x03
#define MCP_RECOVERY_CLOSE_PRINTER      0x05
#define MCP_RECOVERY_OPERATOR_RESET     0x11
#define MCP_RECOVERY_REBOOT_PRINTER     0x12
#define MCP_RECOVERY_REBOOT_PC          0x13
#define MCP_RECOVERY_EXHAUST_PARTS      0x14
#define MCP_RECOVERY_CHECK_ENV          0x21
#define MCP_RECOVERY_CHECK_DEVICE       0x23
#define MCP_RECOVERY_CHECK_PROGRAM      0x29
#define MCP_RECOVERY_UNKNOWN            0x3F


/* Macros for Get each error info from SDK functions return code   */

#define MCP_ERRFLAG(dwRet)  (BYTE)((dwRet & 0x1F000000) >> 24)  /* Get Error flag       */  /* Change 2005/06/08 Ver.1.3.0 */
#define MCP_ERRINFO(dwRet)  (BYTE)((dwRet & 0x00FF0000) >> 16)  /* Get Error info       */
#define MCP_RECOVERY(dwRet) (BYTE)((dwRet & 0x0000FF00) >> 8)   /* Get Recovery Info    */


/*
 *  MCP_GetLastError() return codes
 */

/* Printer condition error          */

#define MCP_ERR_BUSY_INITIALIZING       0x01010001  /* Printer busy. (Initializing)                                             */
#define MCP_ERR_BUSY_PRINTING           0x01010003  /* Printer busy. (Printing)                                                 */
#define MCP_ERR_BUSY_FEEDANDCUT         0x010100F1  /* (* No use for this SDK)  */
#define MCP_ERR_BUSY_UNKNOWN            0x010100FF  /* Printer busy. (Unknown status)                                           */
#define MCP_ERR_NO_DEVICE               0x01020001  /* No printer connected at there.                                           */

/* Printer error (Print aborted.) */ /* Changed 2005/06/08 Ver.1.3.0 */

#define MCP_ERR_NO_PAPER                0x0200000A  /* No Paper.                                */
#define MCP_ERR_NO_INK                  0x0200000B  /* No Ink ribbon.                           */
#define MCP_ERR_PAPER_DOOR_OPEN         0x0200000C  /* Paper door opens.                        */
#define MCP_ERR_INK_DOOR_OPEN           0x0200000D  /* Ink door opens.                          */
#define MCP_ERR_MEDIA_MISMATCH          0x0200000E  /* Illegal size paper is set.               */
#define MCP_ERR_PAPER_JAM               0x02000014  /* Paper JAM.                               */
#define MCP_ERR_INK_JAM                 0x02000015  /* Ink ribbon JAM.                          */
#define MCP_ERR_MECHA_ERROR             0x02000016  /* Mechanical error.                        */

/* Printer error (Print continues.) */ /* Add 2005/06/08 Ver.1.3.0 */

#define MCP_ERR_PRINTING_PAPER_END          0x1200000A  /* Paper ended. (Print continues.)              */
#define MCP_ERR_PRINTING_INK_END            0x1200000B  /* Ink ribbon ended. (Print continues.)         */
#define MCP_ERR_PRINTING_PAPER_DOOR_OPEN    0x1200000C  /* Paper door opens. (Print continues.)         */
#define MCP_ERR_PRINTING_INK_DOOR_OPEN      0x1200000D  /* Ink door opens. (Print continues.) Note: "DPB-4000 Printer does not return."   */
#define MCP_ERR_PRINTING_MEDIA_MISMATCH     0x1200000E  /* Illegal size paper is set.(Print continues.) */


/* Communication interface error    */

#define MCP_ERR_USB_CONTROL             0x03010101  /* USB control error.                                                       */

/* Software error                   */

#define MCP_ERR_NO_CONTROL              0x04010001  /* No method of printer control.                                            */
#define MCP_ERR_NO_IMGSET_1DLUT_DATA    0x04010152  /* Failed to load 1D LUT data file.                                         */  /* 2005/06/24 Ver.1.4.0     */
#define MCP_ERR_NO_IMGSET_3DLUT_DATA    0x04010153  /* Failed to load 3D LUT data file in same folder of this DLL.              */
#define MCP_ERR_NO_INIT                 0x04010201  /* Yet not call MCP_Init().                                                 */
#define MCP_ERR_STILL_OPEN              0x04010202  /* There is printer which is still opened by call MCP_OpenPrinter().        */
#define MCP_ERR_ALREADY_OPEN            0x04010203  /* Printer Indicated at MCP_OpenPrinter() is already opened.                */
#define MCP_ERR_MAX_OPEN                0x04010204  /* Already max count printers open.                                         */
#define MCP_ERR_NO_OPEN                 0x04010205  /* Printer Indicated is not opened.                                         */
#define MCP_ERR_INVALID_HPRINTER        0x04010301  /* Printer handle is invalid.                                               */
#define MCP_ERR_INVALID_PARAMETER       0x04010302  /* Function parameter is invalid.                                           */
#define MCP_ERR_IMAGE_OUTOF_PAPER       0x04010303  /* The image position is out of paper.                                      */
#define MCP_ERR_NOTSUPPORT_IMAGE_FMT    0x04010304  /* Image format is not 24Bpp DIB.                                           */
#define MCP_ERR_ALREADY_START_THREAD    0x04010305  /* Already max count of thread for print run for target printer.            */
#define MCP_ERR_UNMATCH_IMAGESIZE       0x04010306  /* Image size is un-match to set.                                           */
#define MCP_ERR_MEMORY_ALLOC            0x04020001  /* Failed to allocate memory.                                               */
#define MCP_ERR_FAIL_START_THREAD       0x04020002  /* Failed to start thread.                                                  */

#define MCP_ERR_UNKNOWN                 0x04FFFFFF  /* Unknown error.                                                           */


/*
 *  Bitmask for MCP_GetPrinterError() return value( dwPrinterErrorDetail ) //  2006/03/13 Ver.2.2.0
 */ 
#define MCP_BM_NUM                         21
#define MCP_BM_MEDIA_MISMATCH      0x00000001
#define MCP_BM_PAPER_JAM           0x00000002
#define MCP_BM_RIBBON_END          0x00000004
#define MCP_BM_PAPER_END           0x00000008
#define MCP_BM_RIBBONDOOR_OPEN     0x00000010
#define MCP_BM_RIBBONMARK_ERROR    0x00000020
#define MCP_BM_PAPERDOOR_OPEN      0x00000040
#define MCP_BM_MOTOR_SENSOR_ERROR  0x00000080
#define MCP_BM_NO_RIBBON           0x00000100
#define MCP_BM_NO_PAPER            0x00000200
#define MCP_BM_SYSTEM_ERROR        0x00000400
#define MCP_BM_HEAD_HEATING        0x00000800
#define MCP_BM_HEAD_COOLING        0x00001000
#define MCP_BM_BUFFER_EMPTY        0x00002000
#define MCP_BM_BUFFER_USED         0x00004000
#define MCP_BM_BUFFER_FULL         0x00008000
#define MCP_BM_PARAMETER_ERROR     0x00010000
#define MCP_BM_COMMAND_ERROR       0x00020000
#define MCP_BM_TRANSFER_ERROR      0x00040000
#define MCP_BM_USB_ERROR           0x00080000
#define MCP_BM_RECOVERY_PROCESSING 0x00100000


#ifdef __cplusplus
}
#endif


#endif /* #ifndef __DPBAPI_H__ */
