/************************************************************/
/*                                                          */
/*  FUJIFILM Image Setup Process Library			        */
/*                                                          */
/*  File Name:  FFISPL.h		                            */
/*                                                          */
/*      Last Modified at 2006/03/31						    */
/*                                                          */
/*  Copyright (C) Fuji Photo Film Co., Ltd.			        */
/*  All rights reserved.                                    */
/*                                                          */
/************************************************************/

#ifndef _FFISPL_H_
#define _FFISPL_H_

#if !defined(ISPLFUNC)
#define ISPLFUNC __declspec(dllimport)
#endif

#ifdef __cplusplus
extern "C" {
#endif  // __cplusplus

// Error Code
typedef enum emISPL_ERR_CODE
{
	ISPL_NOERR					=	0,		// Normal termination.
	ISPL_ERR_LOAD_DLL			=	1001,	// Failed to load I2 DLL.
	ISPL_ERR_PARAM_INIT			=	1002,	// ISPL_Init() is not called.
	ISPL_ERR_PARAM_FINISH		=	1003,	// ISPL_Finish() is not called.
	ISPL_ERR_MANUAL_SETUP		=	1004,	// ISPL_ImgSetupWithRedEye() is not called.
	ISPL_ERR_SETUP_TYPE			=	1005,	// Invalid value for setup type.
	ISPL_ERR_NOT_FOUND_FILE		=	1006,	// Not exist the specified file.
	ISPL_ERR_CREATE_FILE		=	1007,	// Failed to create the file.
	ISPL_ERR_DELETE_FILE		=	1008,	// Failed to delete the file.
	ISPL_ERR_READ_FILE			=	1009,	// Failed to read the file.
	ISPL_ERR_WRITE_FILE			=	1010,	// Failed to write the file.
	ISPL_ERR_PARAM_FOLDER		=	1011,	// The file opening in the Param folder failed.
	ISPL_ERR_SCN_PROFILE		=	1012,	// The scanner profile is NULL.
	ISPL_ERR_PRN_PROFILE		=	1013,	// The type of the printer profile is not correct.

	ISPL_ERR_REDEYE_CLASS_NEW	=	1101,	// The class creation of red-eye failed.
	ISPL_ERR_REDEYE_MEM_NEW		=	1102,	// Failed to allocate memory of red-eye.
	ISPL_ERR_REDEYE_SETDETIMG	=	1103,	// The input image creation processing of red-eye failed.
	ISPL_ERR_REDEYE_DETECT		=	1104,	// The detection processing of red-eye failed.
	ISPL_ERR_REDEYE_CONVDETDAT	=	1105,	// The information conversion processing of red-eye failed.
	ISPL_ERR_REDEYE_CORRECT		=	1106,	// The correction processing of red-eye failed.
	ISPL_ERR_REDEYE_PROCMODE	=	1107,	// The rectangular area outside the range was specified.
	ISPL_ERR_REDEYE_RECTAREA	=	1108,	// The processing mode outside the range was specified.
	ISPL_ERR_REDEYE_UNDO		=	1109,	// The specified red-eye removal has already been returned.

	ISPL_ERR_COLOR_PARAM		=	1201,	// Invalid value for color(C, M, Y) correction.
	ISPL_ERR_BRIGHTNESS_PARAM	=	1202,	// Invalid value for brightness correction.
	ISPL_ERR_CONTRAST_PARAM		=	1203,	// Invalid value for contrast correction.
	ISPL_ERR_MONOTONE_PARAM		=	1204,	// The monotone parameter is illegal.
	ISPL_ERR_COLCHROMA_PARAM	=	1205,	// Invalid value for color chroma correction.
	ISPL_ERR_CHROMAVAL_PARAM	=	1206,	// Invalid value for chroma correction.
	ISPL_ERR_SHARPNESS_PARAM	=	1207,	// Invalid value for sharpness strength.
	ISPL_ERR_BEAUTSKIN_PARAM	=	1208,	// Invalid value for beautiful skin strength.
	ISPL_ERR_NOISERED_PARAM		=	1209,	// Invalid value for noise reduction strength.
	ISPL_ERR_READ_EXIFTAG		=	1300,	// Failed to read tag information from the file.
	ISPL_ERR_GET_EXIFTAG_SIZE	=	1301,	// Failed to get the size of tag information.
	ISPL_ERR_WRITE_EXIFTAG		=	1302,	// Failed to write tag information to memory.

	ISPL_ERR_MEM_NEW			=	2001,	// Failed to allocate memory.
	ISPL_ERR_NULL_PTR			=	2002,	// Parameter is NULL.
	ISPL_ERR_NOT_SUPPORT_EXT	=	2004,	// An unsupported file extension.
	ISPL_ERR_FILE_FORMAT		=	2005,	// This file format has not been permitted.
	ISPL_ERR_FORMAT_PARAM		=	2009,	// Invalid parameter for image format.
	ISPL_ERR_BOOL_PARAM			=	2010,	// The truth value parameter outside the range was specified.
	ISPL_ERR_NOT_BMP			=	2018,	// BMP(DIB) data not correct.
	ISPL_ERR_NOT_SUPPORT_IMAGE	=	2019,	// The image data not supported in this API. (depth, channel, and compression type, etc.)
	ISPL_ERR_INSIZE_WH			=	2020,	// An input image size outside the range. (It is too small.)
	ISPL_ERR_SETUP_PARAM		=	2037,	// The setup parameter is illegal.
	ISPL_ERR_LEVEL_PARAM		=	2039,	// The level parameter is illegal.
	ISPL_ERR_UNKNOWN			=	9999,	// Unkown error
} ISPL_ERR_CODE;

#ifdef UNIEDIT
// Information for image data 
typedef struct{
	LPVOID lpConvertImg;		// Image data for conversion (DIB)
	LPVOID lpOriginalImg;		// Original image data (DIB)
	int nSetupType;				// Setup type
	PWCHAR pImgFilePath;		// Image file name (full path)
} ISPL_IMAGE_INFO, *PISPL_IMAGE_INFO;

// Information for profile 
typedef struct{
	PWCHAR pScannerProfile;		// Scanner profile file name (full path)
	PWCHAR pPrinterProfile;		// Printer profile file name (full path)
} ISPL_PROFILE_INFO, *PISPL_PROFILE_INFO;

#else
// Information for image data 
typedef struct{
	LPVOID lpConvertImg;		// Image data for conversion (DIB)
	LPVOID lpOriginalImg;		// Original image data (DIB)
	int nSetupType;				// Setup type
	PCHAR pImgFilePath;			// Image file name (full path)
} ISPL_IMAGE_INFO, *PISPL_IMAGE_INFO;

// Information for profile 
typedef struct{
	PCHAR pScannerProfile;		// Scanner profile file name (full path)
	PCHAR pPrinterProfile;		// Printer profile file name (full path)
} ISPL_PROFILE_INFO, *PISPL_PROFILE_INFO;
#endif

// Parameters for image setup processing
typedef struct {
	short sColorC;				// C density correction value
	short sColorM;				// M density correction value
	short sColorY;				// Y density correction value
	short sBrightness;			// Brightness correction value
	short sContrast;			// Contrast correction value
	int nMonotoneMode;			// Monotone mode
	short sColChroma;			// Color chroma correction value
	short sChromaVal;			// Chroma correction value
	short sSharpnessPower;		// Sharpness power value
	BOOL bBeautSkinSwitch;		// Beautiful skin switch
	short sBeautSkinPower;		// Beautiful skin power value
	BOOL bNoiseRedSwitch;		// Noise reduction switch
	short sNoiseRedPower;		// Noise reduction power value
} ISPL_MANUAL_PARAM, *PISPL_MANUAL_PARAM;

// Area information after the red-eye removal.
typedef struct {
	LONG lnPointX;				// Peak coordinates X
	LONG lnPointY;				// Peak coordinates Y
	LONG lnRectX;				// Coordinates X on the left
	LONG lnRectY;				// Coordinates Y on the left
	ULONG ulRectW;				// Rectangular horizontal direction
	ULONG ulRectH;				// Rectangular vertical direction
} ISPL_REDEYE_AREA, *PISPL_REDEYE_AREA;

// Result of the red-eye removal for one face.
typedef struct {
	ISPL_REDEYE_AREA RedEyeArea[2];		// Area information on one face
} ISPL_REDEYE_FACEDATA, *PISPL_REDEYE_FACEDATA;

// Total number of the red-eye removed faces and result of each face.
typedef struct {
	int nFaceNum;							// Total number of removed faces
	PISPL_REDEYE_FACEDATA pRedEyeFaceData;	// Result for removed faces
} ISPL_REDEYE_RESULT, *PISPL_REDEYE_RESULT;

// Information to the red-eye removal (processing mode and rectangular area).
typedef struct {
	BYTE byProcMode;			// Processing mode of the red-eye removal 
	LONG lnRectX;				// Coordinates X on the left
	LONG lnRectY;				// Coordinates Y on the left
	ULONG ulRectW;				// Rectangular horizontal direction
	ULONG ulRectH;				// Rectangular vertical direction
} ISPL_REDEYE_MANUALINFO, *PISPL_REDEYE_MANUALINFO;

#define		ISPL_AUTOSETUP_OFF		(0)		// Auto image setup processing OFF
#define		ISPL_AUTOSETUP_ON		(1)		// Auto image setup processing ON

#define		ISPL_REDEYE_OFF			(0)		// Red-eye removal OFF
#define		ISPL_REDEYE_ON			(1)		// Red-eye removal ON

#define		ISPL_REDEYE_MODE_MANUAL	(0)		// Red-eye removal mode Manual 
#define		ISPL_REDEYE_MODE_CANCEL	(1)		// Red-eye removal mode Cancel 

#define		ISPL_SETUP_TYPE_DSC		(0)		// Setup type for DSC
#define		ISPL_SETUP_TYPE_SCN		(1)		// Setup type for Scanner

#define		ISPL_MONOTONE_NONE		(0)		// Monotone OFF
#define		ISPL_MONOTONE_BW		(1)		// Monotone mode black and white
#define		ISPL_MONOTONE_F_SEPIA	(2)		// Monotone mode fuji sepia

#define		ISPL_BEAUTSKIN_OFF		(0)		// Beautiful skin OFF
#define		ISPL_BEAUTSKIN_ON		(1)		// Beautiful skin ON

#define		ISPL_NOISERED_OFF		(0)		// Noise reduction OFF
#define		ISPL_NOISERED_ON		(1)		// Noise reduction ON


// Initialize the parameters.
int	ISPLFUNC ISPL_Init();

// Terminate the parameters.
int	ISPLFUNC ISPL_Finish();

// Set the default values to the parameters.
int	ISPLFUNC ISPL_SetDefaultParam(PISPL_MANUAL_PARAM);

// Execute the automatic image setup processing and the red-eye automatic removal.
int	ISPLFUNC ISPL_ImgSetupWithRedEye(BOOL, BOOL, ISPL_IMAGE_INFO, ISPL_MANUAL_PARAM, 
									 PISPL_REDEYE_RESULT, ISPL_PROFILE_INFO);

// Execute the manual image setup processing.
int	ISPLFUNC ISPL_ManualSetup(ISPL_IMAGE_INFO, ISPL_MANUAL_PARAM);

// Execute the red-eye automatic removal.
int	ISPLFUNC ISPL_RedEye(ISPL_IMAGE_INFO, PISPL_REDEYE_RESULT);

// Execute the red-eye manual removal.
int ISPLFUNC ISPL_RedEyeManual(ISPL_IMAGE_INFO, int, PISPL_REDEYE_MANUALINFO, PISPL_REDEYE_RESULT);

// Return the image before executing previous ISPL_RedEye() or ISPL_RedEyeManual().
int ISPLFUNC ISPL_RedEyeUndo(ISPL_IMAGE_INFO, PISPL_REDEYE_RESULT);

// Convert the color space from sRGB to printer color space.
int	ISPLFUNC ISPL_PrinterProfile(ISPL_IMAGE_INFO, ISPL_PROFILE_INFO);


#ifdef __cplusplus
}
#endif

#endif // #ifndef _FFISPL_H_
