/************************************************************/
/*															*/
/*	File :	Printer.cpp										*/
/*															*/
/*	Printer SDK Sample Program								*/
/*															*/
/*	Copyright (C) 2001-2006 Fuji Photo Film Co., Ltd.		*/
/*															*/
/************************************************************/


#include "stdafx.h"
#include "sdksample.h"
#include "Printer.h"
#include "sdksetup.h"
#include "ErrDlg.h"

#include "Encode.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif


///////////////////////////////////////////////////////////////////////////////
//
//	CPrinter class public method
//


// Constructor / Destructor
// (PrinterNo : Index of target printer got at enumerated CSetup::Enum().
//  hPrinter  : Handle of target printer got at enumerated CSetup::Enum().
//  strProductName : Printer product name.)
CPrinter::CPrinter(int PrinterNo, HANDLE hPrinter, LPCTSTR strProductName)
{

	m_fOpen		= FALSE;
	m_printerNo	= PrinterNo;
	m_hPrinter	= hPrinter;
	m_strProductName = strProductName;
	m_fRecoverableError = FALSE;
	m_printMethod = PRINTMETHOD_NORMAL;
	m_printMode = MCP_PRINT_MODE_SPEED;	// nakashima addition 2010/01/18 Ver.2.7.1
}

CPrinter::~CPrinter()
{

}


// Start controlling target printer.
// (Return value : TRUE OK, FALSE error)
BOOL CPrinter::Open()
{
	DWORD	ret;

	// Start controlling target printer.
	ret = fnSDK.OpenPrinter(m_hPrinter, MCP_TIMEOUT_INFINITE);
	if (ret != MCP_OK) {
		ErrMessage(_T("MCP_OpenPrinter()"), ret);
		return	FALSE;
	}
	m_fOpen = TRUE;

	return	TRUE;
}


// Finish controlling target printer.
// (Return value : TRUE OK, FALSE error)
BOOL CPrinter::Close()
{
	DWORD	ret;

	// Finish controlling target printer.
	ret = fnSDK.ClosePrinter(m_hPrinter);
	if (ret != MCP_OK) {
		ErrMessage(_T("MCP_ClosePrinter()"), ret);
		return	FALSE;
	}
	m_fOpen = FALSE;

	return	TRUE;
}


// Get device information of target printer.
// (pPrinterInfo : [OUT] Device information.)
// (Return value : TRUE OK, FALSE error)
BOOL CPrinter::GetPrinterInfo(PMCP_PRINTER_INFO pPrinterInfo)
{
	DWORD	ret;

	// Get device information of target printer.
	ret = fnSDK.GetPrinterInfo(m_hPrinter, pPrinterInfo);
	if (ret != MCP_OK) {
		ErrMessage(_T("MCP_GetPrinterInfo()"), ret);
		return	FALSE;
	}

	return	TRUE;
}


// Clear frame memory of target printer.
// (Return value : TRUE OK, FALSE error)
BOOL CPrinter::PrinterMemoryClear()
{
	DWORD		ret;

	// Clear printer frame memory of target printer.
	ret = fnSDK.InitPrinterMemory(m_hPrinter);
	if (ret != MCP_OK) {
		ErrMessage(_T("MCP_InitPrinterMemory()"), ret);
		return	FALSE;
	}

	// Initialize SDK printsetting.
	ret = fnSDK.InitPrintSetting(m_hPrinter);
	if (ret != MCP_OK) {
		ErrMessage(_T("MCP_InitPrintSetting()"), ret);
		return	FALSE;
	}

	return	TRUE;
}


// Send print setting to target printer.
// (PrintInfo : Print setting.
//  pfPrinterCannotReceiveDataNow : [OUT] TRUE = Printer is printing and cannot receive image data now.)
// (Return value : TRUE OK, FALSE error (include pfPrinterCannotReceiveData is TRUE) )
BOOL CPrinter::SendPrintSetting(PRINT_INFO PrintInfo, PBOOL pfPrinterCannotReceiveDataNow)
{
	DWORD		ret;
	DWORD		errCode;

	*pfPrinterCannotReceiveDataNow = FALSE;

	// Send print setting to taget printer by SDK setting.
	ret = fnSDK.SendPrintSetting(m_hPrinter, PrintInfo.PrintSetting);
	if (ret != MCP_OK) {
		errCode = fnSDK.GetLastError(m_hPrinter);
		if (errCode == MCP_ERR_BUSY_PRINTING) {
			// Printer is printing and cannot receive image data now.
			*pfPrinterCannotReceiveDataNow = TRUE;
		} else {
			ErrMessage(_T("MCP_SendPrintSetting()"), ret);
		}
		return	FALSE;
	}

	return	TRUE;
}


// Send image data to printer, and start print, and resister callback function for notification printing to SDK.
// (hDib      : Image DIB memory handle.
//  pCallback :	Address of callback function for SDK notification printing.)
// (Return value : TRUE OK, FALSE error)
BOOL CPrinter::SendImageAndPrint(HANDLE hDib, PMCP_CALLBACK pCallback)
{
	DWORD		ret;

	// Sending image data during printing.
	ret = fnSDK.SendImageAndPrint(m_hPrinter, hDib, pCallback);
	if (ret != MCP_OK) {
		ErrMessage(_T("MCP_SendImageAndPrint()"), ret);
		return	FALSE;
	}
	// MCP_SendImageAndPrint() function returns quickly because make thread for print.

	return	TRUE;
}

// Send image data to printer, and start print, and resister callback function for notification printing to SDK.
// (hDib      : 1st Image DIB memory handle.
// (hDib2     : 2nd Image DIB memory handle.
//  pCallback :	Address of callback function for SDK notification printing.)
// (Return value : TRUE OK, FALSE error)
BOOL CPrinter::SendImageAndMultiPrint(HANDLE hDib, HANDLE hDib2, PMCP_CALLBACK pCallback)
{
	DWORD		ret;

	// Sending image data during printing.
	ret = fnSDK.SendImageAndMultiPrint(m_hPrinter, hDib, hDib2, pCallback);
	if (ret != MCP_OK) {
		ErrMessage(_T("MCP_SendImageAndMultiPrint()"), ret);
		return	FALSE;
	}
	// MCP_SendImageAndMultiPrint() function returns quickly because make thread for print.

	return	TRUE;
}

#ifndef RET_VER247	// nakashima addition 2007/06/29 Ver.2.5.0
// Send image data to printer, and start print, and resister callback function for notification printing to SDK.
// (hDib      : 1st Image DIB memory handle.
// (hDib2     : 2nd Image DIB memory handle.
//  pCallback :	Address of callback function for SDK notification printing.)
// (Return value : TRUE OK, FALSE error)
BOOL CPrinter::SendImageAndThreeDivPrint(HANDLE hDib, HANDLE hDib2, HANDLE hDib3, PMCP_CALLBACK pCallback)
{
	DWORD		ret;

	// Sending image data during printing.
	ret = fnSDK.SendImageAndThreeDivPrint(m_hPrinter, hDib, hDib2, hDib3, pCallback);
	if (ret != MCP_OK) {
		ErrMessage(_T("MCP_SendImageAndThreeDivPrint()"), ret);
		return	FALSE;
	}
	// MCP_SendImageAndThreeDivPrint() function returns quickly because make thread for print.

	return	TRUE;
}
#endif

// Cancel print in the way.
// (Return value : TRUE OK, FALSE error)
BOOL CPrinter::CancelPrint()
{
	DWORD		ret;


	// Cancel multipul print in the way.
	ret = fnSDK.CancelPrint(m_hPrinter);
	if (ret != MCP_OK) {
		ErrMessage(_T("MCP_CancelPrint()"), ret);
		return	FALSE;
	}

	return	TRUE;
}


// Stop print.
// (Return value : TRUE OK, FALSE error)
BOOL CPrinter::StopPrint()
{
	DWORD		ret;

	ret = fnSDK.StopPrint(m_hPrinter);
	if (ret != MCP_OK) {
		ErrMessage(_T("MCP_StopPrint()"), ret);
		return	FALSE;
	}

	return	TRUE;
}


// Get printer remain paper (Copies).
// (pRemainCopies : [OUT] Remain copies.)
// (Return value : TRUE OK, FALSE error)
BOOL CPrinter::GetPaperRemain(PDWORD pRemainCopies)
{
	DWORD		ret;

	// Get printer remain paper.
	ret = fnSDK.GetPaperRemain(m_hPrinter, pRemainCopies);
	if (ret != MCP_OK) {
		ErrMessage(_T("MCP_GetPaperRemain()"), ret);
		return	FALSE;
	}

	return	TRUE;
}


#ifndef RET_VERSION_265		// nakashima addition 2008/02/10 Ver.2.6.6
// Get printer info (TotalPrintedCnt, TotalHeadPrintedCnt).
// (pTotalPrintedCnt : [OUT] Total printed count after shipment.)
// (pTotalHeadPrintedCnt : [OUT] Total printed count after head change.)
// (Return value : TRUE OK, FALSE error)
BOOL CPrinter::GetPrintedInfo(PDWORD pTotalPrintedCnt, PDWORD pTotalHeadPrintedCnt)
{
	DWORD		ret;

	// Get total printed count after shipment.
	ret = fnSDK.ReceivePrintedInfo(m_hPrinter, pTotalPrintedCnt);
	if (ret != MCP_OK) {
		ErrMessage(_T("MCP_ReceivePrintedInfo()"), ret);
		return	FALSE;
	}

	// Get total printed count after head change.
	ret = fnSDK.ReceiveHeadPrintedInfo(m_hPrinter, pTotalHeadPrintedCnt);
	if (ret != MCP_OK) {
		ErrMessage(_T("MCP_ReceiveHeadPrintedInfo()"), ret);
		return	FALSE;
	}

	return	TRUE;
}
#endif

// Execute 3D LUT process to image data.
// (hDib            : Image DIB memory handle. (Original & Result of process)
//  strDataFileName : File name of 3D LUT parameter data file. (File name only (Not full-path.))
// (Return value : TRUE OK, FALSE error)
BOOL CPrinter::Do3dLut(HANDLE hDib, PCSTR strDataFileName)
{
	DWORD		ret;

	// Execute 3D LUT process to image data.
	ret = fnSDK.LutConversionForPrinter(hDib, strDataFileName);
	if (ret != MCP_OK) {
		ErrMessage(_T("MCP_LutConversionForPrinter()"), ret, TRUE);
		return	FALSE;
	}
	return	TRUE;
}

/**/
#ifndef RET_VERSION_250		// nakashima addition 2007/08/09 Ver.2.6.0
BOOL CPrinter::Create1dDataFileOldLinear(int iD, int iY, int iM, int iC, CString& strDataPath)
#else
BOOL CPrinter::Create1dDataFile(int iD, int iY, int iM, int iC, CString& strDataPath)
#endif
{
	DWORD		ret;
	BOOL	bRet = TRUE;

	unsigned char* lpData = NULL;

	char	szFilePath[MAX_PATH];
	char	szDrive[_MAX_DRIVE], szDir[_MAX_DIR];

	GetModuleFileNameA( NULL, szFilePath, _MAX_PATH );
	_splitpath( szFilePath, szDrive, szDir, NULL, NULL );
	sprintf(szFilePath, "%s%s", szDrive, szDir);	

	lpData = (unsigned char *)GlobalAlloc( GPTR, 256*3*sizeof(unsigned char) );

	ret = fnSDK.Generate1dData(lpData, iD, iC, iM, iY);
	if (ret != MCP_OK) {
		ErrMessage(_T("MCP_Generate1dData()"), ret, TRUE);
		bRet = FALSE;
		goto Exit;
	}

	
	ret = fnSDK.Create1dDataFile(m_hPrinter, lpData, szFilePath);
	strDataPath = szFilePath;
	if (ret != MCP_OK) {
		ErrMessage(_T("MCP_Create1dDataFile()"), ret, TRUE);
		bRet = FALSE;
		goto Exit;
	}

Exit:
	if(lpData!=NULL){
		GlobalFree(lpData);
	}
	return	bRet;
}

/**/
/* Ver.2.4.0 */
#ifndef RET_VERSION_250		// nakashima addition 2007/08/09 Ver.2.6.0
BOOL CPrinter::Create1dDataFileOldGamma(int iD, int iY, int iM, int iC, CString& strDataPath)
#else
BOOL CPrinter::Create1dDataFileWithGammaCurve(int iD, int iY, int iM, int iC, CString& strDataPath)
#endif
{
	DWORD		ret;
	BOOL	bRet = TRUE;

	unsigned char* lpData = NULL;

	char	szFilePath[MAX_PATH];
	char	szDrive[_MAX_DRIVE], szDir[_MAX_DIR];

	GetModuleFileNameA( NULL, szFilePath, _MAX_PATH );
	_splitpath( szFilePath, szDrive, szDir, NULL, NULL );
	sprintf(szFilePath, "%s%s", szDrive, szDir);	

	lpData = (unsigned char *)GlobalAlloc( GPTR, 256*3*sizeof(unsigned char) );

	ret = fnSDK.Generate1dDataWithGammaCurve(lpData, iD, iC, iM, iY);
	if (ret != MCP_OK) {
		ErrMessage(_T("MCP_Generate1dDataWithGammaCurve()"), ret, TRUE);
		bRet = FALSE;
		goto Exit;
	}

	
	ret = fnSDK.Create1dDataFile(m_hPrinter, lpData, szFilePath);
	strDataPath = szFilePath;
	if (ret != MCP_OK) {
		ErrMessage(_T("MCP_Create1dDataFile()"), ret, TRUE);
		bRet = FALSE;
		goto Exit;
	}

Exit:
	if(lpData!=NULL){
		GlobalFree(lpData);
	}
	return	bRet;
}

#ifndef RET_VERSION_250		// nakashima addition 2007/08/09 Ver.2.6.0
BOOL CPrinter::Create1dDataFileNewLinear(int iR, int iG, int iB, CString& strDataPath)
{
	DWORD		ret;
	BOOL	bRet = TRUE;

	unsigned char* lpData = NULL;

	char	szFilePath[MAX_PATH];
	char	szDrive[_MAX_DRIVE], szDir[_MAX_DIR];

	GetModuleFileNameA( NULL, szFilePath, _MAX_PATH );
	_splitpath( szFilePath, szDrive, szDir, NULL, NULL );
	sprintf(szFilePath, "%s%s", szDrive, szDir);	

	lpData = (unsigned char *)GlobalAlloc( GPTR, 256*3*sizeof(unsigned char) );

	ret = fnSDK.Make1dDataLinear(lpData, iR, iG, iB);
	if (ret != MCP_OK) {
		ErrMessage(_T("MCP_Make1dDataLinear()"), ret, TRUE);
		bRet = FALSE;
		goto Exit;
	}

	
	ret = fnSDK.Create1dDataFile(m_hPrinter, lpData, szFilePath);
	strDataPath = szFilePath;
	if (ret != MCP_OK) {
		ErrMessage(_T("MCP_Create1dDataFile()"), ret, TRUE);
		bRet = FALSE;
		goto Exit;
	}

Exit:
	if(lpData!=NULL){
		GlobalFree(lpData);
	}
	return	bRet;
}

BOOL CPrinter::Create1dDataFileNewGamma(int iR, int iG, int iB, CString& strDataPath)
{
	DWORD		ret;
	BOOL	bRet = TRUE;

	unsigned char* lpData = NULL;

	char	szFilePath[MAX_PATH];
	char	szDrive[_MAX_DRIVE], szDir[_MAX_DIR];

	GetModuleFileNameA( NULL, szFilePath, _MAX_PATH );
	_splitpath( szFilePath, szDrive, szDir, NULL, NULL );
	sprintf(szFilePath, "%s%s", szDrive, szDir);	

	lpData = (unsigned char *)GlobalAlloc( GPTR, 256*3*sizeof(unsigned char) );

	ret = fnSDK.Make1dDataGammaCurve(lpData, iR, iG, iB);
	if (ret != MCP_OK) {
		ErrMessage(_T("MCP_Make1dDataGammaCurve()"), ret, TRUE);
		bRet = FALSE;
		goto Exit;
	}

	
	ret = fnSDK.Create1dDataFile(m_hPrinter, lpData, szFilePath);
	strDataPath = szFilePath;
	if (ret != MCP_OK) {
		ErrMessage(_T("MCP_Create1dDataFile()"), ret, TRUE);
		bRet = FALSE;
		goto Exit;
	}

Exit:
	if(lpData!=NULL){
		GlobalFree(lpData);
	}
	return	bRet;
}
#endif

/**/
BOOL CPrinter::Delete1dDataFile(){
	DWORD	ret;
	DWORD	errCode;
	char	szFilePath[MAX_PATH];
	char	szDrive[_MAX_DRIVE], szDir[_MAX_DIR];

	GetModuleFileNameA( NULL, szFilePath, _MAX_PATH );
	_splitpath( szFilePath, szDrive, szDir, NULL, NULL );
	sprintf(szFilePath, "%s%s", szDrive, szDir);	

	ret = fnSDK.Delete1dDataFile(m_hPrinter, szFilePath);
	if (ret != MCP_OK) {
		errCode = fnSDK.GetLastError(NULL);

		if( errCode == MCP_ERR_NO_IMGSET_1DLUT_DATA ){
			AfxMessageBox(IDS_ERR_SDKS_NO_IMGSET_1DLUT_DATA, MB_OK | MB_ICONSTOP);		
			return	FALSE;
		}
		else{
			ErrMessage(_T("MCP_Exec1dLutConversion()"), ret, TRUE);
			return	FALSE;
		}	
	}

	return	TRUE;
}


BOOL CPrinter::Do1dLut(HANDLE hDib){
	DWORD	ret = MCP_OK;
//	DWORD	errCode;

	// Ver.1.6.0 2005/08/24 Not used in DPB-1000 Printer 
	if (m_strProductName != "DPB-1000") {
		ret = fnSDK.Exec1dConversion(m_hPrinter, hDib);
	}
	if (ret != MCP_OK) {
		// 2005/12/06 Ver.1.7.1 Notice All Error
		ErrMessage(_T("MCP_Exec1dLutConversion()"), ret, TRUE);
		return	FALSE;

/*		errCode = fnSDK.GetLastError(NULL);

		if( errCode != MCP_ERR_NO_IMGSET_1DLUT_DATA ){
			ErrMessage(_T("MCP_Exec1dLutConversion()"), ret, TRUE);
			return	FALSE;
		}	*/
	}
	return	TRUE;
}

// Display error message. (Indicate string or resource ID of StringTable)
// (Str      : Error message string.
//  StrID    : Error message string ID in StringTable.
//  ErrCode  : SDK functions return code.
//             (If not SDK function error, set 0.)
//  fSdkAllErr : SDK error to all printer ==> TRUE.
void CPrinter::ErrMessage(LPCTSTR Str, DWORD ErrCode, BOOL fSdkAllErr)
{
	CString		errStr;
	DWORD		lastErr;
	int			result;

	char		PrinterError[4];
	DWORD		PrinterStatus;

	if (ErrCode != 0) {
		// Get SDK last error.
		if (fSdkAllErr) {
			lastErr = fnSDK.GetLastError(NULL);
		} else {
			lastErr = fnSDK.GetLastError(m_hPrinter);
			fnSDK.GetPrinterError(m_hPrinter, PrinterError, &PrinterStatus);
		}

		// Display SDK error dialog.
		CErrDlg errDlg;
		// 2006/03/13 Ver.2.3.0
//		errDlg.SetErrorInfo(Str, m_printerNo, ErrCode, lastErr);
		errDlg.SetErrorInfo(Str, m_printerNo, ErrCode, lastErr, 0, PrinterError, PrinterStatus);
		result = errDlg.DoModal();
	} else {
		// Only display MessageBox.
		errStr.Format(_T("%s\n\tPrinter No. = %02d"), Str, m_printerNo);
		AfxMessageBox(errStr, MB_OK | MB_ICONSTOP);
	}

}

void CPrinter::ErrMessage(UINT StrID, DWORD ErrCode)
{
	CString		str;

	str.LoadString(StrID);

	CPrinter::ErrMessage(str, ErrCode);

}

// Check whether error is recoverable or not.
// (Return value : TRUE yes, FALSE no)
// Add 2005/6/9 mochida
BOOL CPrinter::IsRecoverable()
{
	BOOL	bRet;
	DWORD	lastErr;
	lastErr = fnSDK.GetLastError(m_hPrinter);

	bRet = MCP_IS_ERRF_H_CONTINUE(lastErr);
	m_fRecoverableError = bRet;

	return bRet;
}


#ifndef RET_VERSION_270	// nakashima addition 2010/01/18 Ver.2.7.1
// 2005/08/26 Ver.1.6.1
// Table for 3D LUT parameter data file name of each printer.
typedef struct {
	TCHAR		ProductName[MAX_PATH];		// Product name
	int			nPrintMode;					// Print mode
	char		szDataFileName[MAX_PATH];	// File name of 3D LUT parameter data file.
} LUT3D_FILE_NAME_INFO, *PLUT3D_FILE_NAME_INFO;

static LUT3D_FILE_NAME_INFO	Lut3dFileNameTbl[] = {
	{ _T("Unknown"),	MCP_PRINT_MODE_SPEED,	"",	},
	{ _T("DPB-1000"),	MCP_PRINT_MODE_SPEED,	"",	},
	{ _T("ASK-1500"),	MCP_PRINT_MODE_SPEED,	"DP3D1500.dat",	},
	{ _T("ASK-4000"),	MCP_PRINT_MODE_SPEED,	"DP3D4000.dat",	},
	{ _T("ASK-2000"),	MCP_PRINT_MODE_SPEED,	"DP3D2000.dat",	},	// 2005/12/26 Ver.2.0.3
	{ _T("ASK-4000A"),	MCP_PRINT_MODE_SPEED,	"DP3D4000A.dat",},	// nakashima addition 2007/06/29 Ver.2.5.0
	{ _T("ASK-2500"),	MCP_PRINT_MODE_SPEED,	"DP3D2500.dat",	},	// takanaka addition 2008/10/09 Ver.2.6.8
	{ _T("ASK-2000"),	MCP_PRINT_MODE_QUALITY,	"DP3D2000HQ.dat",	},	// nakashima addition 2010/01/18 Ver.2.7.1
	{ _T("ASK-2500"),	MCP_PRINT_MODE_QUALITY,	"DP3D2500HQ.dat",	},	// nakashima addition 2010/01/18 Ver.2.7.1
};
#else
// 2005/08/26 Ver.1.6.1
// Table for 3D LUT parameter data file name of each printer.
typedef struct {
	TCHAR		ProductName[MAX_PATH];		// Product name
	char		szDataFileName[MAX_PATH];	// File name of 3D LUT parameter data file.
} LUT3D_FILE_NAME_INFO, *PLUT3D_FILE_NAME_INFO;

static LUT3D_FILE_NAME_INFO	Lut3dFileNameTbl[] = {
	{ _T("Unknown"),	"",	},
	{ _T("DPB-1000"),	"",	},
	{ _T("ASK-1500"),	"DP3D1500.dat",	},
	{ _T("ASK-4000"),	"DP3D4000.dat",	},
	{ _T("ASK-2000"),	"DP3D2000.dat",	},	// 2005/12/26 Ver.2.0.3
	{ _T("ASK-4000A"),	"DP3D4000A.dat",},	// nakashima addition 2007/06/29 Ver.2.5.0
	{ _T("ASK-2500"),	"DP3D2500.dat",	},	// takanaka addition 2008/10/09 Ver.2.6.8
};
#endif

// Execute 3D LUT process to image data.
// (hDib            : Image DIB memory handle. (Original & Result of process)
// (Return value : TRUE OK, FALSE error)
BOOL CPrinter::Do3dLut(HANDLE hDib) // 2005/08/26 Ver.1.6.1
{
	DWORD		ret;
	char		szDataFileName[MAX_PATH];

	// Execute 3D LUT process to image data.
	// Ver.1.6.0 2005/08/24 Not used in DPB-1000 Printer 
	for (int i = 0; i < COUNTOF_ARRAY(Lut3dFileNameTbl); i++) {
#ifndef RET_VERSION_270	// nakashima addition 2010/01/18 Ver.2.7.1
		if ((m_strProductName == Lut3dFileNameTbl[i].ProductName)
		&& (m_printMode == Lut3dFileNameTbl[i].nPrintMode)){
#else
		if (m_strProductName == Lut3dFileNameTbl[i].ProductName){
#endif
			break;
		}
	}

	strcpy( szDataFileName, Lut3dFileNameTbl[i].szDataFileName); 

	// Execute 3D LUT process to image data.
	if (szDataFileName != "") {
		ret = fnSDK.LutConversionForPrinter(hDib, szDataFileName);
		if (ret != MCP_OK) {
			ErrMessage(_T("MCP_LutConversionForPrinter()"), ret, TRUE);
			return	FALSE;
		}
	}
	return	TRUE;
}

// Set AutoRecovery Flag. 2005/12/08 Ver.1.8.0
BOOL CPrinter::SetAutoRecoveryFlag(BOOL fAuto){
	DWORD		ret;

	m_fAutoRecovery = fAuto;
	ret = fnSDK.SetAutoRecoveryFlag(m_hPrinter, fAuto);
	if (ret != MCP_OK) {
		ErrMessage(_T("MCP_SetAutoRecoveryFlag()"), ret, TRUE);
		return	FALSE;
	}
	return	TRUE;
	
}

// 2006/03/13 Ver.2.3.0
BOOL CPrinter::GetPrinterError(HANDLE hPrinter, char *szPrinterError, DWORD *dwPrinterErrorDetail)
{
	DWORD		ret;

	ret = fnSDK.GetPrinterError(m_hPrinter, szPrinterError, dwPrinterErrorDetail);
	if (ret != MCP_OK) {
		ErrMessage(_T("MCP_GetPrinterError()"), ret, TRUE);
		return	FALSE;
	}
	return	TRUE;	
}

///////////////////////////////////////////////////////////////////////////////
//
//	CPrinter class private method
//



